/**
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.solr.response;

import java.io.OutputStream;
import java.io.IOException;
import java.io.Writer;

import org.apache.solr.common.SolrDocumentList;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.request.SolrQueryRequest;

import org.apache.solr.response.BaseResponseWriter.SingleResponseWriter; // javadocs

/**
 * 
 * 
 * A generic {@link QueryResponseWriter} implementation that requires a user to
 * implement the
 * {@link #getSingleResponseWriter(OutputStream, SolrQueryRequest, SolrQueryResponse)}
 * that defines a {@link SingleResponseWriter} to handle the binary output.
 * 
 * @since 1.5
 * @version $Id: GenericBinaryResponseWriter.java 1004075 2010-10-04 00:36:21Z rmuir $
 * 
 */
public abstract class GenericBinaryResponseWriter extends BaseResponseWriter
    implements BinaryQueryResponseWriter {

  /**
   * 
   * Writes the binary output data using the {@link SingleResponseWriter}
   * provided by a call to
   * {@link #getSingleResponseWriter(OutputStream, SolrQueryRequest, SolrQueryResponse)}
   * .
   * 
   * @param out
   *          The {@link OutputStream} to write the binary data to.
   * @param request
   *          The provided {@link SolrQueryRequest}.
   * @param response
   *          The provided {@link SolrQueryResponse}.
   */
  public void write(OutputStream out, SolrQueryRequest request,
      SolrQueryResponse response) throws IOException {
    super.write(getSingleResponseWriter(out, request, response), request,
        response);
  }

  /**
   * Users of this class should implement this method to define a
   * {@link SingleResponseWriter} responsible for writing the binary output
   * given a {@link SolrDocumentList} or doc-by-doc, given a
   * {@link SolrInputDocument}.
   * 
   * @param out
   *          The {@link OutputStream} to write the binary data response to.
   * @param request
   *          The provided {@link SolrQueryRequest}.
   * @param response
   *          The provided {@link SolrQueryResponse}.
   * @return A {@link SingleResponseWriter} that will be used to generate the
   *         response output from this {@link QueryResponseWriter}.
   */
  public abstract SingleResponseWriter getSingleResponseWriter(
      OutputStream out, SolrQueryRequest request, SolrQueryResponse response);

  /**Just to throw Exception So that the eimplementing classes do not have to do the  same
   */
  public void write(Writer writer, SolrQueryRequest request, SolrQueryResponse response) throws IOException {
    throw new RuntimeException("This is a binary writer , Cannot write to a characterstream");
  }
}
