/*
 * Copyright (C) 2016 The Flogger Authors.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package com.google.common.flogger.testing;

import static com.google.common.truth.Truth.assertAbout;
import static com.google.common.truth.Truth.assertWithMessage;

import com.google.common.flogger.backend.FormatType;
import com.google.common.truth.FailureMetadata;
import com.google.common.truth.Subject;
import javax.annotation.Nullable;

/**
 * A <a href="https://github.com/google/truth">Truth</a> subject for {@link FormatType}.
 *
 * @author Kurt Alfred Kluever (kak@google.com)
 */
public final class FormatTypeSubject extends Subject<FormatTypeSubject, FormatType> {

  public static FormatTypeSubject assertThat(@Nullable FormatType formatType) {
    return assertAbout(FormatTypeSubject.FORMAT_TYPE_SUBJECT_FACTORY).that(formatType);
  }

  private static final Subject.Factory<FormatTypeSubject, FormatType> FORMAT_TYPE_SUBJECT_FACTORY =
      FormatTypeSubject::new;

  private FormatTypeSubject(FailureMetadata failureMetadata, @Nullable FormatType subject) {
    super(failureMetadata, subject);
  }

  public void canFormat(Object arg) {
    assertWithMessage("Unable to format " + arg + " using " + getSubject())
        .that(getSubject().canFormat(arg))
        .isTrue();
  }

  public void cannotFormat(Object arg) {
    assertWithMessage("Expected error when formatting " + arg + " using " + getSubject())
        .that(getSubject().canFormat(arg))
        .isFalse();
  }

  public void isNumeric() {
    assertWithMessage("Expected " + getSubject() + " to be numeric but wasn't")
        .that(getSubject().isNumeric())
        .isTrue();
  }

  public void isNotNumeric() {
    assertWithMessage("Expected " + getSubject() + " to not be numeric but was")
        .that(getSubject().isNumeric())
        .isFalse();
  }
}
