/**
 * Copyright (c) 2004-2006 Regents of the University of California.
 * See "license-prefuse.txt" for licensing terms.
 */
package prefuse.visual.tuple;

import java.util.Iterator;

import prefuse.data.Graph;
import prefuse.data.Table;
import prefuse.data.expression.Predicate;
import prefuse.data.util.FilterIterator;
import prefuse.visual.AggregateItem;
import prefuse.visual.AggregateTable;
import prefuse.visual.VisualItem;


/**
 * AggregateItem implementation that uses data values from a backing
 * AggregateTable.
 * 
 * @author <a href="http://jheer.org">jeffrey heer</a>
 */
public class TableAggregateItem extends TableVisualItem 
    implements AggregateItem
{   
    /**
     * Initialize a new TableAggregateItem for the given table and row. This
     * method is used by the appropriate TupleManager instance, and should not
     * be called directly by client code, unless by a client-supplied custom
     * TupleManager.
     * @param table the data Table
     * @param graph ignored by this class
     * @param row the table row index
     */
    protected void init(Table table, Graph graph, int row) {
        this.m_table = table;
        this.m_row = this.m_table.isValidRow(row) ? row : -1;
    }

    /**
     * @see prefuse.visual.AggregateItem#getAggregateSize()
     */
    public int getAggregateSize() {
        return ((AggregateTable)this.m_table).getAggregateSize(this.m_row);
    }

    /**
     * @see prefuse.visual.AggregateItem#containsItem(prefuse.visual.VisualItem)
     */
    public boolean containsItem(VisualItem item) {
        return ((AggregateTable)this.m_table).aggregateContains(this.m_row, item);
    }

    /**
     * @see prefuse.visual.AggregateItem#addItem(prefuse.visual.VisualItem)
     */
    public void addItem(VisualItem item) {
        ((AggregateTable)this.m_table).addToAggregate(this.m_row, item);
    }

    /**
     * @see prefuse.visual.AggregateItem#removeItem(prefuse.visual.VisualItem)
     */
    public void removeItem(VisualItem item) {
        ((AggregateTable)this.m_table).removeFromAggregate(this.m_row, item);
    }

    /**
     * @see prefuse.visual.AggregateItem#removeAllItems()
     */
    public void removeAllItems() {
        ((AggregateTable)this.m_table).removeAllFromAggregate(this.m_row);
    }

    /**
     * @see prefuse.visual.AggregateItem#items()
     */
    public Iterator items() {
        return ((AggregateTable)this.m_table).aggregatedTuples(this.m_row);
    }
    
    /**
     * @see prefuse.visual.AggregateItem#items()
     */
    public Iterator items(Predicate filter) {
        return new FilterIterator(
            ((AggregateTable)this.m_table).aggregatedTuples(this.m_row), filter);
    }

} // end of class TableAggregateItem
