package org.apache.cassandra.utils;
/*
 * 
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 * 
 *   http://www.apache.org/licenses/LICENSE-2.0
 * 
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 * 
 */


import java.net.InetAddress;
import java.nio.ByteBuffer;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.util.UUID;

import org.safehaus.uuid.EthernetAddress;
import org.safehaus.uuid.UUIDGenerator;

/**
 * Generates type 1 (time-based) UUIDs
 */
public class UUIDGen
{
    /** creates a type1 uuid but substitutes hash of the IP where the mac would go. */
    public static synchronized UUID makeType1UUIDFromHost(InetAddress addr)
    {
        try
        {
            MessageDigest digest = MessageDigest.getInstance("MD5");
            digest.update(addr.getAddress());
            byte[] md5 = digest.digest();
            byte[] fauxMac = new byte[6];
            System.arraycopy(md5, 0, fauxMac, 0, Math.min(md5.length, fauxMac.length));
            return getUUID(ByteBuffer.wrap(UUIDGenerator.getInstance().generateTimeBasedUUID(new EthernetAddress(fauxMac)).toByteArray()));
        }
        catch (NoSuchAlgorithmException ex)
        {
            throw new RuntimeException("Your platform has no support for generating MD5 sums");
        }
    }
    
    /** creates a type 1 uuid from raw bytes. */
    public static UUID getUUID(ByteBuffer raw)
    {
        return new UUID(raw.getLong(raw.position()), raw.getLong(raw.position() + 8));
    }

    /** decomposes a uuid into raw bytes. */
    public static byte[] decompose(UUID uuid)
    {
        long most = uuid.getMostSignificantBits();
        long least = uuid.getLeastSignificantBits();
        byte[] b = new byte[16];
        for (int i = 0; i < 8; i++)
        {
            b[i] = (byte)(most >>> ((7-i) * 8));
            b[8+i] = (byte)(least >>> ((7-i) * 8));
        }
        return b;
    }
}
